/*
 *  							PS/2 Keyer
 *
 *  Copyright (C) 2009  David Bern, W2LNX     W2LNX@ARRL.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 *  USA, or see <http://www.gnu.org/licenses/>.
 */

#include "device.h"
#include "configuration.h"
#include "constants.h"
#include "interrupts.h"

/******************************************************************************/
/*																			  */
/*				interrupt management and handling routines					  */
/*																			  */			
/******************************************************************************/

/* -------------------------------------------------------------------------- */

enum {		/* constants */

	TICK_COUNT 		= 0xF63A,	/* 10 msec tick interval at 8 MHz clock */
	// TICK_COUNT 	= 0xE7A8,	/* 10 msec tick interval at 20 MHz clock */
	TICK_TIME 	    = 10,		/* 10 msec */
};

/* -------------------------------------------------------------------------- */

static unsigned int Port_B = 0;
static unsigned int Interrupt_event = NO_INTERRUPT;
static unsigned long Timer1_countdown = 0;
static unsigned long Timer3_countdown = 0;

/* -------------------------------------------------------------------------- */

/*
 * this routine initializes the interrupts
 */

void intialize_interrupts(void)
{
	int enabled;

#if TESTING == 1
	printf("initialize_interrupts():\r\n");

	printf("    TICK_COUNT:      %Lu\r\n"
		   "    TICK_TIME:       %Ld msec\r\n",  0xFFFF - TICK_COUNT, TICK_TIME);
#endif

	enabled = INTERRUPTS_ENABLED;
	if (enabled == FALSE) {
#if TESTING == 1
		printf("initialize_interrupts() skipped\r\n");
#endif
		return;
	}

	enabled = INT_EXT_ENABLED;
	if (enabled == TRUE) {
		ext_int_edge(0, H_TO_L);
		enable_interrupts(INT_EXT);
	}

	enabled = INT_EXT1_ENABLED;
	if (enabled == TRUE) {
		ext_int_edge(1, H_TO_L);		
		enable_interrupts(INT_EXT1);
	}

	enabled = INT_EXT2_ENABLED;
	if (enabled == TRUE) {
		ext_int_edge(2, H_TO_L);		
		enable_interrupts(INT_EXT2);
	}

	enabled = INT_RB_ENABLED;
	if (enabled == TRUE) {
		enable_interrupts(INT_RB);
	}

	enabled = INT_TIMER1_ENABLED;
	if (enabled == TRUE) {	
		setup_timer_1(T1_INTERNAL | T1_DIV_BY_8);
		enable_interrupts(INT_TIMER1);
	}

	enabled = INT_TIMER3_ENABLED;
	if (enabled == TRUE) {	
		setup_timer_3(T3_INTERNAL | T3_DIV_BY_8);
		enable_interrupts(INT_TIMER3);
	}

	enable_interrupts(GLOBAL);

#if TESTING == 1
	printf("initialize_interrupts() done\r\n");
#endif
}

/* -------------------------------------------------------------------------- */

/*
 * this routine returns the interrupt event
 */

unsigned int get_interrupt_event(void)
{

	return Interrupt_event;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine clears the interrupt event
 */

void clear_interrupt_event(void)
{

	Interrupt_event = NO_INTERRUPT;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine sets up a timer in milliseconds
 */

void set_timer_ms(unsigned long time)
{

	Timer3_countdown = time + TICK_TIME;
	set_timer3(TICK_COUNT);
}
/* -------------------------------------------------------------------------- */

/*
 * this function returns the value of port B read by the rb_isr routine
 */

unsigned int get_port_B(void)
{

	return Port_B;
}

/* ========================================================================== */

/*
 * this routine is the change on pins B4:B7 interrupt service routine
 */

#INT_RB
void rb_isr(void)
{

	Port_B = input_B();		/* clears the interrupt */
	Interrupt_event = RB_INTERRUPT;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine is the 16 bit timer1 overflow interrupt service routine
 */

#INT_TIMER1
void timer1_isr(void)
{

	set_timer1(TICK_COUNT);

	if (Timer1_countdown > TICK_TIME) {		
		Timer1_countdown -= TICK_TIME;
	} else if (Timer1_countdown == TICK_TIME) {
		Timer1_countdown = 0;
		Interrupt_event = TIMER1_INTERRUPT;
	}
}

/* -------------------------------------------------------------------------- */

/*
 * this routine is the 16 bit timer3 overflow interrupt service routine
 */

#INT_TIMER3
void timer3_isr(void)
{

	set_timer3(TICK_COUNT);

	if (Timer3_countdown > TICK_TIME) {		
		Timer3_countdown -= TICK_TIME;
	} else if (Timer3_countdown == TICK_TIME) {
		Timer3_countdown = 0;
		Interrupt_event = TIMER3_INTERRUPT;
	}
}

/* -------------------------------------------------------------------------- */

/*
 * this routine is the pin B0 interrupt service routine
 */

#INT_EXT
void ext_isr(void)
{

	Interrupt_event = EXT_INTERRUPT;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine is the pin B1 interrupt service routine
 */

#INT_EXT1
void ext1_isr(void)
{

	Interrupt_event = EXT1_INTERRUPT;
}

/* -------------------------------------------------------------------------- */

/*
 * this routine is the pin B2 interrupt service routine
 */

#INT_EXT2
void ext2_isr(void)
{

	Interrupt_event = EXT2_INTERRUPT;
}

/* -------------------------------------------------------------------------- */
